/*
 * @(#)StructParser.java  1.3  2005-01-15
 *
 * Copyright (c) 2000-2005 Werner Randelshofer
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */
package ch.randelshofer.media.binary;

import java.io.*;
import ch.randelshofer.util.ParseException;
import ch.randelshofer.util.*;
import java.util.*;
import java.text.*;

/**
 * Parses structured binary data using C-like data declarations.
 *
 * Syntax:
 * <pre><code>
 * Declarations ::= { MagicDeclaration | DescriptionDeclaration | EnumDeclaration | SetDeclaration | TypedefDeclaration } EOF
 * MagicDeclaration ::= "magic" identifier stringLiteral ";"
 * DescriptionDeclaration ::= "description" identifier stringLiteral "," stringLiteral ";"
 * EnumDeclaration ::= EnumSpecifier identifier ";"
 * SetDeclaration ::= SetSpecifier identifier ";"
 * TypedefDeclaration ::= "typedef" TypeSpecifier identifier ";"
 * EnumSpecifier ::= "enm" ( identifier | "{" identifier ["=" magicOrIntLiteral] {"," identifier ["=" intLiteral]} "}" )
 * SetSpecifier ::= "set" ( identifier | "{" identifier "=" intLiteral {"," identifier "=" intLiteral} "}" )
 * TypeSpecifier ::= ( StructSpecifier | (PrimitiveSpecifier [EnumSpecifier | SetSpecifier]) ) [ArrayList]
 * StructSpecifier ::= "struct (identifier | "{" MemberDeclaration {"," MemberDeclaration } "}" )
 * MemberDeclaration ::= TypeSpecifier identifier [ArrayList] ";"
 * PrimitiveSpecifier ::= "byte" | "short" | "ushort" | "int" | "long" | "float" | "double" | "extended"
 *                         | "char" | "cstring" | "pstring" | "magic" | "mactimestamp"
 * ArrayList ::= "[" [ArraySize] "]" {"," identifier "[" [ArraySize] "]" }
 * ArraySize ::= () | intLiteral ["-" intLiteral]
 * </code></pre>
 * @author Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Swityerland
 * @version 1.3 2005-01-09 Support for little-endian data types added. Support
 * for magic enmerations added.
 * <br>1.2.1 2003-04-01 Revised.
 * <br>1.2  2000-10-04 Support for set's added.
 * <br> 1.1  2000-10-03 Support for enm's added.
 * <br> 1.0 2000-06-11
 */
public class StructParser extends Object {
    private Declarations declarations;
    
    public StructParser() {
    }
    public StructParser(Reader r) throws IOException, ParseException {
        parse(r);
    }
    
    public String getName(String magic) {
        MagicDeclaration md = (MagicDeclaration) declarations.magics.get(magic);
        DescriptionDeclaration dd = (md == null) ? null : (DescriptionDeclaration) declarations.descriptions.get(md.identifier);
        return (dd == null) ? null : dd.name;
    }
    public boolean isMagicDeclared(String magic) {
        MagicDeclaration md = (MagicDeclaration) declarations.magics.get(magic);
        return md != null;
    }
    
    public String getDescription(String magic) {
        MagicDeclaration md = (MagicDeclaration) declarations.magics.get(magic);
        DescriptionDeclaration dd = (md == null) ? null : (DescriptionDeclaration) declarations.descriptions.get(md.identifier);
        return (dd == null) ? null : dd.description;
    }
    
    public StructTableModel readStruct(String magic, byte[] data)
    throws IOException {
        return declarations.readStruct(magic, data);
    }
    
    private static String errorMsg(String text, StreamTokenizer scanner) {
        StringBuffer b = new StringBuffer();
        b.append(Integer.toString(scanner.lineno()));
        b.append(':');
        b.append(text);
        b.append(" instead of ");
        switch (scanner.ttype) {
            case StreamTokenizer.TT_WORD :
                b.append('\'');
                b.append(scanner.sval);
                b.append('\'');
                break;
            case StreamTokenizer.TT_EOF :
                b.append("EOF");
                break;
            case StreamTokenizer.TT_EOL :
                b.append("EOL");
                break;
            case StreamTokenizer.TT_NUMBER :
                b.append(Double.toString(scanner.nval));
                break;
            case '"' :
                b.append('"');
                b.append(scanner.sval);
                b.append('"');
                break;
            default :
                b.append('\'');
                b.append((char) scanner.ttype);
                b.append('\'');
                break;
        }
        return b.toString();
    }
    
    protected void parse(Reader r)
    throws IOException, ParseException {
        StreamTokenizer scanner = new StreamTokenizer(r);
        scanner.resetSyntax();
        scanner.wordChars('a', 'z');
        scanner.wordChars('A', 'Z');
        scanner.wordChars(128 + 32, 255);
        scanner.wordChars('_', '_');
        scanner.whitespaceChars(0, ' ');
        //scanner.commentChar('/');
        scanner.quoteChar('"');
        scanner.quoteChar('\'');
        scanner.parseNumbers();
        scanner.slashSlashComments(true);
        scanner.slashStarComments(true);
        scanner.ordinaryChar(';');
        scanner.ordinaryChar('[');
        scanner.ordinaryChar(']');
        scanner.ordinaryChar('{');
        scanner.ordinaryChar('}');
        
        declarations = new Declarations(scanner);
    }
    
    /**
     * Declarations expression.
     *
     * <pre><code>
     * Declarations ::= { MagicDeclaration | DescriptionDeclaration | EnumDeclaration | SetDeclaration | TypedefDeclaration }
     * </code></pre>
     */
    protected static class Declarations {
        public Hashtable magics = new Hashtable();
        public Hashtable descriptions = new Hashtable();
        public Hashtable enms = new Hashtable();
        public Hashtable sets = new Hashtable();
        public Hashtable typedefs = new Hashtable();
        
        public Declarations(StreamTokenizer scanner)
        throws IOException, ParseException {
            while (scanner.nextToken() != StreamTokenizer.TT_EOF) {
                if (scanner.ttype == StreamTokenizer.TT_WORD) {
                    if (scanner.sval.equals("magic")) {
                        scanner.pushBack();
                        MagicDeclaration md = new MagicDeclaration(scanner);
                        magics.put(md.magic, md);
                    } else if (scanner.sval.equals("description")) {
                        scanner.pushBack();
                        DescriptionDeclaration dd = new DescriptionDeclaration(scanner);
                        descriptions.put(dd.identifier, dd);
                    } else if (scanner.sval.equals("enm")) {
                        scanner.pushBack();
                        EnumDeclaration ed = new EnumDeclaration(scanner);
                        enms.put(ed.identifier, ed);
                    } else if (scanner.sval.equals("set")) {
                        scanner.pushBack();
                        SetDeclaration sd = new SetDeclaration(scanner);
                        sets.put(sd.identifier, sd);
                    } else if (scanner.sval.equals("typedef")) {
                        scanner.pushBack();
                        TypedefDeclaration td = new TypedefDeclaration(scanner);
                        typedefs.put(td.identifier, td);
                    } else {
                        throw new ParseException(errorMsg("Declarations: Expected 'magic', 'description', 'enm', 'set' or 'typedef' ",scanner));
                    }
                }
            }
        }
        public StructTableModel readStruct(String magic, byte[] data)
        throws IOException {
            Vector result = new Vector();
            TypedefDeclaration typedef = null;
            
            DataInputStream in = new DataInputStream(new ByteArrayInputStream(data));
            MagicDeclaration magicdef = (MagicDeclaration) magics.get(magic);
            if (magicdef == null) {
                //System.out.println("StructParser.StructTableModel.readStruct() unknow magic:"+magic);
                //new Exception().printStackTrace();
                throw new IOException("unknown magic:"+magic);
            }
            typedef = (TypedefDeclaration) typedefs.get(magicdef.identifier);
            if (typedef == null) {
                //System.out.println("StructParser.StructTableModel.readStruct() typedef not found for magic:"+magic+" magicdef.identifier:"+magicdef.identifier);
                throw new IOException("unknown type:"+magicdef.identifier);
            }
            try {
                Object obj = typedef.read(in, this, result);
                if (obj != null) {
                    StructTableModel.Value value = new StructTableModel.Value();
                    value.declaration = typedef.identifier;
                    value.value = obj;
                    result.addElement(value);
                }
            } catch (EOFException e) {}
            return new StructTableModel(typedef, result);
        }
    }
    
    /**
     * MagicDeclaration expression.
     *
     * <pre><code>
     * MagicDeclaration ::= "magic" identifier stringLiteral ";"
     * </code></pre>
     */
    protected static class MagicDeclaration {
        public String identifier;
        public String magic;
        
        public MagicDeclaration(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() != StreamTokenizer.TT_WORD || ! scanner.sval.equals("magic")) {
                throw new ParseException(errorMsg("MagicDeclaration: 'magic' expected",scanner));
            }
            if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                throw new ParseException(errorMsg("MagicDeclaration: identifier expected",scanner));
            }
            identifier = scanner.sval;
            if (scanner.nextToken() != '"') {
                throw new ParseException(errorMsg("MagicDeclaration: string literal expected",scanner));
            }
            magic = scanner.sval;
            if (scanner.nextToken() != ';') {
                throw new ParseException(errorMsg("MagicDeclaration: ';' expected",scanner));
            }
        }
    }
    /**
     * DescriptionDeclaration expression.
     *
     * <pre><code>
     * DescriptionDeclaration ::= "description" identifier stringLiteral "," stringLiteral ";"
     * </code></pre>
     */
    protected static class DescriptionDeclaration {
        public String identifier;
        public String name;
        public String description;
        
        public DescriptionDeclaration(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() != StreamTokenizer.TT_WORD || ! scanner.sval.equals("description")) {
                throw new ParseException(errorMsg("DescriptionDeclaration: 'magic' expected",scanner));
            }
            if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                throw new ParseException(errorMsg("DescriptionDeclaration: identifier expected",scanner));
            }
            identifier = scanner.sval;
            if (scanner.nextToken() != '"') {
                throw new ParseException(errorMsg("DescriptionDeclaration: string literal (title) expected",scanner));
            }
            name = scanner.sval;
            if (scanner.nextToken() != ',') {
                throw new ParseException(errorMsg("DescriptionDeclaration: ',' expected",scanner));
            }
            if (scanner.nextToken() != '"') {
                throw new ParseException(errorMsg("DescriptionDeclaration: string literal (description) expected",scanner));
            }
            description = scanner.sval;
            if (scanner.nextToken() != ';') {
                throw new ParseException(errorMsg("DescriptionDeclaration: ';' expected",scanner));
            }
        }
    }
    /**
     * EnumDeclaration expression.
     *
     * <pre><code>
     * EnumDeclaration ::= EnumSpecifier identifier ";"
     * </code></pre>
     */
    protected static class EnumDeclaration {
        public EnumSpecifier enmSpecifier;
        public String identifier;
        
        public EnumDeclaration(StreamTokenizer scanner)
        throws IOException, ParseException {
            enmSpecifier = new EnumSpecifier(scanner);
            
            if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                throw new ParseException(errorMsg("EnumDeclaration: identifier expected",scanner));
            }
            identifier = scanner.sval;
            
            if (scanner.nextToken() != ';') {
                throw new ParseException(errorMsg("EnumDeclaration: ';' expected",scanner));
            }
        }
    }
    /**
     * SetDeclaration expression.
     *
     * <pre><code>
     * SetDeclaration ::= SetSpecifier identifier ";"
     * </code></pre>
     */
    protected static class SetDeclaration {
        public SetSpecifier setSpecifier;
        public String identifier;
        
        public SetDeclaration(StreamTokenizer scanner)
        throws IOException, ParseException {
            setSpecifier = new SetSpecifier(scanner);
            
            if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                throw new ParseException(errorMsg("SetDeclaration: identifier expected",scanner));
            }
            identifier = scanner.sval;
            
            if (scanner.nextToken() != ';') {
                throw new ParseException(errorMsg("SetDeclaration: ';' expected",scanner));
            }
        }
    }
    /**
     * TypedefDeclaration expression.
     *
     * <pre><code>
     * TypedefDeclaration ::= "typedef" TypeSpecifier identifier ";"
     * </code></pre>
     */
    protected static class TypedefDeclaration {
        public TypeSpecifier typeSpecifier;
        public String identifier;
        
        public TypedefDeclaration(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() != StreamTokenizer.TT_WORD || ! scanner.sval.equals("typedef")) {
                throw new ParseException(errorMsg("TypedefDeclaration: 'typedef' expected",scanner));
            }
            
            typeSpecifier = new TypeSpecifier(scanner);
            
            if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                throw new ParseException(errorMsg("TypedefDeclaration: identifier expected",scanner));
            }
            identifier = scanner.sval;
            
            if (scanner.nextToken() != ';') {
                throw new ParseException(errorMsg("TypedefDeclaration: ';' expected",scanner));
            }
            //System.out.println("typedef "+typeSpecifier+" "+identifier+";");
        }
        public Object read(DataInputStream in, Declarations declarations, Vector result)
        throws IOException {
            return typeSpecifier.read(in, declarations, identifier, result);
        }
    }
    /**
     * EnumSpecifier expression.
     *
     * <pre><code>
     * EnumSpecifier ::= "enm" ( identifier | "{" identifier ["=" intLiteral] {"," identifier ["=" intLiteral]} "}" )
     * </code></pre>
     */
    protected static class EnumSpecifier {
        public Hashtable members;
        public String identifier;
        public boolean isMagicEnum;
        
        public EnumSpecifier(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() != StreamTokenizer.TT_WORD || ! scanner.sval.equals("enm")) {
                throw new ParseException(errorMsg("EnumSpecifier: 'enm' expected",scanner));
            }
            
            if (scanner.nextToken() == StreamTokenizer.TT_WORD) {
                identifier = scanner.sval;
            } else if (scanner.ttype == '{') {
                String name;
                int value = -1;
                members = new Hashtable();
                do {
                    if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                        throw new ParseException(errorMsg("EnumSpecifier: enmeration name expected",scanner));
                    }
                    name = scanner.sval;
                    
                    if (scanner.nextToken() == '=') {
                        MagicOrIntLiteral literal = new MagicOrIntLiteral(scanner);
                        value = literal.intValue();
                        isMagicEnum = literal.isMagic();
                    } else {
                        value++;
                        scanner.pushBack();
                    }
                    
                    members.put(new Integer(value), name);
                } while (scanner.nextToken() == ',');
                if (scanner.ttype != '}') {
                    throw new ParseException(errorMsg("EnumSpecifier: '}' expected",scanner));
                }
            } else  {
                throw new ParseException(errorMsg("EnumSpecifier: identifier or '{' expected",scanner));
            }
            identifier = scanner.sval;
        }
        
        public String toEnumString(int value, Declarations declarations) {
            if (identifier != null) {
                EnumDeclaration enmDecl = (EnumDeclaration) declarations.enms.get(identifier);
                if (enmDecl == null) {
                    throw new InternalError("Enum Declaration missing for "+identifier);
                }
                return enmDecl.enmSpecifier.toEnumString(value, declarations);
            } else {
                StringBuffer buf = new StringBuffer();
                Integer intValue = new Integer(value);
                if (isMagicEnum) {
                    buf.append('"');
                    buf.append(MagicOrIntLiteral.toMagic(value));
                    buf.append('"');
                } else {
                    buf.append("0x");
                    buf.append(Integer.toHexString(value));
                }
                buf.append(" {");
                if (members.get(intValue) != null) {
                    buf.append(members.get(intValue).toString());
                }
                buf.append('}');
                return buf.toString();
            }
        }
        public String toEnumString(String value, Declarations declarations) {
            if (identifier != null) {
                EnumDeclaration enmDecl = (EnumDeclaration) declarations.enms.get(identifier);
                if (enmDecl == null) {
                    throw new InternalError("Enum Declaration missing for "+identifier);
                }
                return enmDecl.enmSpecifier.toEnumString(value, declarations);
                } else {
                if (value == null || value.length() != 4 || ! isMagicEnum) {
                return value;
            } else {
                return toEnumString(MagicOrIntLiteral.toInt(value), declarations);
            }
                }
        }
    }
    
    /**
     * SetSpecifier expression.
     *
     * <pre><code>
     * SetSpecifier ::= "set" ( identifier | "{" identifier "=" intLiteral {"," identifier "=" intLiteral} "}" )
     * </code></pre>
     */
    protected static class SetSpecifier {
        public Hashtable members;
        public String identifier;
        
        public SetSpecifier(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() != StreamTokenizer.TT_WORD || ! scanner.sval.equals("set")) {
                throw new ParseException(errorMsg("SetSpecifier: 'set' expected",scanner));
            }
            
            if (scanner.nextToken() == StreamTokenizer.TT_WORD) {
                identifier = scanner.sval;
            } else if (scanner.ttype == '{') {
                String name;
                int value = 0;
                members = new Hashtable();
                do {
                    
                    if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                        throw new ParseException(errorMsg("SetSpecifier: set name expected",scanner));
                    }
                    name = scanner.sval;
                    
                    if (scanner.nextToken() == '=') {
                        value = (new IntLiteral(scanner)).intValue();
                    } else {
                        value = (value == 0) ? 1 : value << 1;
                        scanner.pushBack();
                    }
                    
                    members.put(new Integer(value), name);
                } while (scanner.nextToken() == ',');
                if (scanner.ttype != '}') {
                    throw new ParseException(errorMsg("SetSpecifier: '}' expected",scanner));
                }
            } else  {
                throw new ParseException(errorMsg("SetSpecifier: identifier or '{' expected",scanner));
            }
            identifier = scanner.sval;
        }
        public String toSetString(int value, Declarations declarations) {
            if (identifier != null) {
                SetDeclaration setDecl = (SetDeclaration) declarations.sets.get(identifier);
                if (setDecl == null) {
                    throw new InternalError("Set Declaration missing for "+identifier);
                }
                return setDecl.setSpecifier.toSetString(value, declarations);
            } else {
                StringBuffer buf = new StringBuffer();
                buf.append("0x");
                buf.append(Integer.toHexString(value));
                buf.append(" {");
                Enumeration enm = members.keys();
                boolean isFirst = true;
                while (enm.hasMoreElements()) {
                    Object key = enm.nextElement();
                    int intKey = ((Number) key).intValue();
                    if ((intKey == 0 && value == 0) || ((intKey & value) == intKey)) {
                        if (isFirst) {
                            isFirst = false;
                        } else {
                            buf.append(", ");
                        }
                        buf.append(members.get(key).toString());
                    }
                }
                buf.append('}');
                return buf.toString();
            }
        }
    }
    /**
     * TypeSpecifier expression.
     *
     * <pre><code>
     * TypeSpecifier ::= ( StructSpecifier | (PrimitiveSpecifier [EnumSpecifier | SetSpecifier]) ) [ArrayList]
     * ArrayList ::= "[" [ArraySize] "]" {"," identifier "[" [ArraySize] "]" }
     * </code></pre>
     */
    protected static class TypeSpecifier {
        public StructSpecifier structSpecifier;
        public PrimitiveSpecifier primitiveSpecifier;
        public EnumSpecifier enmSpecifier;
        public SetSpecifier setSpecifier;
        public Vector arrayList;
        
        public TypeSpecifier(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() == StreamTokenizer.TT_WORD && scanner.sval.equals("struct")) {
                scanner.pushBack();
                structSpecifier = new StructSpecifier(scanner);
            } else {
                scanner.pushBack();
                primitiveSpecifier = new PrimitiveSpecifier(scanner);
                
                scanner.nextToken();
                if (scanner.ttype == StreamTokenizer.TT_WORD && scanner.sval.equals("enm")) {
                    scanner.pushBack();
                    enmSpecifier = new EnumSpecifier(scanner);
                } else if (scanner.ttype == StreamTokenizer.TT_WORD && scanner.sval.equals("set")) {
                    scanner.pushBack();
                    setSpecifier = new SetSpecifier(scanner);
                } else {
                    scanner.pushBack();
                }
            }
            
            // ArrayList Begin
            if (scanner.nextToken() == '[') {
                arrayList = new Vector();
                do {
                    arrayList.addElement(new ArraySize(scanner));
                    if (scanner.nextToken() != ']') {
                        throw new ParseException(errorMsg("MemberDeclaration: ']' expected",scanner));
                    }
                } while (scanner.nextToken() == '[');
                scanner.pushBack();
            } else {
                scanner.pushBack();
            }
            // ArrayList End
        }
        
        public Object read(DataInputStream in, Declarations declarations, String identifier, Vector result)
        throws IOException {
            // try {
            if (structSpecifier != null) {
                if (arrayList != null) {
                    // ArrayList Begin
                    Enumeration enm = arrayList.elements();
                    while (enm.hasMoreElements()) {
                        ArraySize arraySize = (ArraySize) enm.nextElement();
                        int size = arraySize.getArraySize(declarations, result);
                        if (size == ArraySize.REMAINDER) {
                            try {
                                for (int i=0;;i++) {
                                    int loc = result.size();
                                    structSpecifier.read(in, declarations, result);
                                    
                                    for (int j=loc; j < result.size(); j++) {
                                        StructTableModel.Value arrayValue = (StructTableModel.Value) result.elementAt(j);
                                        arrayValue.index = (arrayValue.index == null) ? "["+i+"]" : "["+i+"]"+arrayValue.index;
                                    }
                                }
                            } catch (EOFException e) {}
                        } else {
                            for (int i=0; i < size; i++) {
                                int loc = result.size();
                                structSpecifier.read(in, declarations, result);
                                
                                for (int j=loc; j < result.size(); j++) {
                                    StructTableModel.Value arrayValue = (StructTableModel.Value) result.elementAt(j);
                                    arrayValue.index = (arrayValue.index == null) ? "["+i+"]" : "["+i+"]"+arrayValue.index;
                                }
                            }
                        }
                    }
                    return null;
                } else {
                    structSpecifier.read(in, declarations, result);
                    return null;
                }
            } else {
                if (arrayList != null) {
                    StringBuffer buf = new StringBuffer();
                    buf.append('{');
                    
                    // ArrayList Begin
                    Enumeration enm = arrayList.elements();
                    while (enm.hasMoreElements()) {
                        ArraySize arraySize = (ArraySize) enm.nextElement();
                        int size = arraySize.getArraySize(declarations, result);
                        if (size == ArraySize.REMAINDER) {
                            try {
                                for (int i=0;;i++) {
                                    //FIXME: does not work properly when the primitive value is not really primitive (e.g. a typedef that defines a struct)
                                    if (i > 0) {
                                        buf.append(", ");
                                    }
                                    buf.append(readValue(in, declarations, result).toString());
                                }
                            } catch (EOFException e) {}
                        } else {
                            for (int i=0; i < size; i++) {
                                //FIXME: does not work properly when the primitive value is not really primitive (e.g. a typedef that defines a struct)
                                if (i > 0) {
                                    buf.append(", ");
                                }
                                buf.append(readValue(in, declarations, result).toString());
                            }
                        }
                    }
                    // ArrayList End
                    
                    buf.append('}');
                    return buf.toString();
                } else {
                    return readValue(in, declarations, result);
                }
            }
            /*
            } catch (IOException e) {
System.out.println("StructParser.TypeSpecifier.read() IOException @"+this);
                throw e;
            }*/
        }
        
        private Object readValue(DataInputStream in, Declarations declarations, Vector result)
        throws IOException {
            Object value = primitiveSpecifier.read(in, declarations, result);
            if (value != null) {
                if (value instanceof Number) {
                    int intValue = ((Number) value).intValue();
                    if (enmSpecifier != null) {
                        value = enmSpecifier.toEnumString(intValue, declarations);
                    } else if (setSpecifier != null) {
                        value = setSpecifier.toSetString(intValue, declarations);
                    }
                } else if (value instanceof String) {
                    if (enmSpecifier != null) {
                        value = enmSpecifier.toEnumString((String) value, declarations);
                    }
                }
            }
            return value;
        }
        public String toString() {
            StringBuffer buf = new StringBuffer();
            buf.append("TypeSpecifier ");
            buf.append(""+structSpecifier);
            buf.append(""+primitiveSpecifier);
            buf.append(""+enmSpecifier);
            buf.append(""+setSpecifier);
            buf.append(""+arrayList);
            return buf.toString();
        }
    }
    /**
     * StructSpecifier expression.
     *
     * <pre><code>
     * StructSpecifier ::= "struct (identifier | "{" MemberDeclaration { MemberDeclaration } "}" )
     * </code></pre>
     */
    protected static class StructSpecifier {
        public String identifier;
        public Vector members;
        
        public StructSpecifier(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() != StreamTokenizer.TT_WORD || ! scanner.sval.equals("struct")) {
                throw new ParseException(errorMsg("StructSpecifier: 'struct' expected",scanner));
            }
            
            scanner.nextToken();
            if (scanner.ttype == StreamTokenizer.TT_WORD) {
                identifier = scanner.sval;
            } else if (scanner.ttype == '{') {
                members = new Vector();
                while (scanner.nextToken() != '}') {
                    scanner.pushBack();
                    
                    members.addElement(new MemberDeclaration(scanner));
                }
            } else {
                throw new ParseException(errorMsg("StructSpecifier: identifier or '{' expected",scanner));
            }
        }
        public void read(DataInputStream in, Declarations declarations, Vector result)
        throws IOException {
            try {
                Enumeration enm = members.elements();
                while (enm.hasMoreElements()) {
                    MemberDeclaration aMember = (MemberDeclaration) enm.nextElement();
                    aMember.read(in, declarations, result);
                }
            } catch (IOException e) {
                System.out.println("StructParser.StructSpecifier.read(...) IOException @ "+identifier);
                throw e;
            }
        }
    }
    /**
     * MemberDeclaration expression.
     *
     * <pre><code>
     * MemberDeclaration ::= TypeSpecifier identifier [ArrayList] ";"
     * ArrayList ::= "[" [ArraySize] "]" {"," identifier "[" [ArraySize] "]" }
     * </code></pre>
     */
    protected static class MemberDeclaration {
        public TypeSpecifier typeSpecifier;
        public String identifier;
        public Vector arrayList;
        
        public MemberDeclaration(StreamTokenizer scanner)
        throws IOException, ParseException {
            typeSpecifier = new TypeSpecifier(scanner);
            
            if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                throw new ParseException(errorMsg("MemberDeclaration: identifier expected",scanner));
            }
            identifier = scanner.sval;
            
            // ArrayList Begin
            if (scanner.nextToken() == '[') {
                arrayList = new Vector();
                do {
                    arrayList.addElement(new ArraySize(scanner));
                    if (scanner.nextToken() != ']') {
                        throw new ParseException(errorMsg("MemberDeclaration: ']' expected",scanner));
                    }
                } while (scanner.nextToken() == '[');
                scanner.pushBack();
            } else {
                scanner.pushBack();
            }
            // ArrayList End
            
            if (scanner.nextToken() == ';') {
            } else if (scanner.ttype == '}') {
                scanner.pushBack();
            } else {
                throw new ParseException(errorMsg("MemberDeclaration: ';' expected",scanner));
            }
        }
        public void read(DataInputStream in, Declarations declarations, Vector result)
        throws IOException {
            try {
                if (arrayList != null) {
                    // ArrayList Begin
                    Enumeration enm = arrayList.elements();
                    while (enm.hasMoreElements()) {
                        ArraySize arraySize = (ArraySize) enm.nextElement();
                        int size = arraySize.getArraySize(declarations, result);
                        if (size == ArraySize.REMAINDER) {
                            try {
                                for (int i=0; ; i++) {
                                    int loc = result.size();
                                    Object obj = typeSpecifier.read(in, declarations, identifier, result);
                                    if (obj != null) {
                                        StructTableModel.Value value = new StructTableModel.Value();
                                        value.declaration = identifier;
                                        value.value = obj;
                                        result.addElement(value);
                                    }
                                    for (int j=loc; j < result.size(); j++) {
                                        StructTableModel.Value arrayValue = (StructTableModel.Value) result.elementAt(j);
                                        arrayValue.index = (arrayValue.index == null) ? "["+i+"]" : "["+i+"]"+arrayValue.index ;
                                    }
                                }
                            } catch (EOFException e) {}
                        } else {
                            for (int i=0; i < size; i++) {
                                int loc = result.size();
                                Object obj = typeSpecifier.read(in, declarations, identifier, result);
                                if (obj != null) {
                                    StructTableModel.Value value = new StructTableModel.Value();
                                    value.declaration = identifier;
                                    value.value = obj;
                                    result.addElement(value);
                                }
                                for (int j=loc; j < result.size(); j++) {
                                    StructTableModel.Value arrayValue = (StructTableModel.Value) result.elementAt(j);
                                    arrayValue.index = (arrayValue.index == null) ? "["+i+"]" : "["+i+"]"+arrayValue.index ;
                                }
                            }
                        }
                    }
                    // ArrayList End
                } else {
                    Object obj = typeSpecifier.read(in, declarations, identifier, result);
                    if (obj != null) {
                        StructTableModel.Value value = new StructTableModel.Value();
                        value.declaration = identifier;
                        value.value = obj;
                        result.addElement(value);
                    }
                }
            } catch (IOException e) {
                System.out.println("StructParser.MemberDeclaration.read(...) IOException @ "+identifier);
                throw e;
            }
            
        }
    }
    /**
     * PrimitiveSpecifier expression.
     *
     * <pre><code>
     * PrimitiveSpecifier ::= "byte" | "short" | "ushort" | "int" | "long" | "float" | "double" | "extended"
     *                         | "char" | "cstring" | "pstring" | "magic" | "mactimestamp"
     * </code></pre>
     */
    protected static class PrimitiveSpecifier {
        private final static char[] HEX = { '0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f' };
        
        /* 8 bits signed. */
        public final static int BYTE = 0;
        /* 8 bits unsigned. */
        public final static int UBYTE = 1;
        /* 16 bits signed big endian. */
        public final static int SHORT = 2;
        /* 16 bits unsigned big endian. */
        public final static int USHORT = 3;
        /* 32 bits signed big endian. */
        public final static int INT = 4;
        /* 32 bits unsigned big endian. */
        public final static int UINT = 5;
        /* 64 bits signed big endian. */
        public final static int LONG = 6;
        /* 32 bits IEEE 754. */
        public final static int FLOAT = 7;
        /* 64 bits IEEE 754. */
        public final static int DOUBLE = 8;
        /* 80 bits IEEE 754. */
        public final static int EXTENDED = 9;
        /* 16 bits UTF-16. */
        public final static int CHAR = 10;
        /* C-Style Zero terminated ASCII String. */
        public final static int CSTRING = 11;
        /* Pascal Style String. Starting with an 8 or 16 bit long length value. */
        public final static int PSTRING = 12;
        /* 4 ASCII characters in 4 subsequent bytes. */
        public final static int MAGIC = 13;
        /* 32 bit signed count of seconds since 1904. */
        public final static int MAC_TIMESTAMP = 14;
        /* A typedef type. */
        public final static int TYPEDEF_TYPE = 15;
        /* 16 bits signed little endian. */
        public final static int SHORTLE = 16;
        /* 16 bits unsigned little endian. */
        public final static int USHORTLE = 17;
        /* 32 bits signed little endian. */
        public final static int INTLE = 18;
        /* 32 bits unsigned little endian. */
        public final static int UINTLE = 19;
        /* 64 bits signed little endian. */
        public final static int LONGLE = 20;
        public int type;
        public String typedef;
        
        public PrimitiveSpecifier(StreamTokenizer scanner)
        throws IOException, ParseException {
            
            if (scanner.nextToken() != StreamTokenizer.TT_WORD) {
                throw new ParseException(errorMsg("PrimitiveSpecifier: primitive type name expected",scanner));
            }
            
            if (scanner.sval.equals("byte")) {
                type = BYTE;
            } else if (scanner.sval.equals("ubyte")) {
                type = UBYTE;
            } else if (scanner.sval.equals("short")) {
                type = SHORT;
            } else if (scanner.sval.equals("ushort")) {
                type = USHORT;
            } else if (scanner.sval.equals("int")) {
                type = INT;
            } else if (scanner.sval.equals("uint")) {
                type = UINT;
            } else if (scanner.sval.equals("long")) {
                type = LONG;
            } else if (scanner.sval.equals("float")) {
                type = FLOAT;
            } else if (scanner.sval.equals("double")) {
                type = DOUBLE;
            } else if (scanner.sval.equals("extended")) {
                type = EXTENDED;
            } else if (scanner.sval.equals("char")) {
                type = CHAR;
            } else if (scanner.sval.equals("cstring")) {
                type = CSTRING;
            } else if (scanner.sval.equals("pstring")) {
                type = PSTRING;
            } else if (scanner.sval.equals("magic")) {
                type = MAGIC;
            } else if (scanner.sval.equals("mactimestamp")) {
                type = MAC_TIMESTAMP;
            } else if (scanner.sval.equals("shortLE")) {
                type = SHORTLE;
            } else if (scanner.sval.equals("ushortLE")) {
                type = USHORTLE;
            } else if (scanner.sval.equals("intLE")) {
                type = INTLE;
            } else if (scanner.sval.equals("uintLE")) {
                type = UINTLE;
            } else if (scanner.sval.equals("longLE")) {
                type = LONGLE;
            } else {
                type = TYPEDEF_TYPE;
                typedef = scanner.sval;
            }
        }
        public Object read(DataInputStream in, Declarations declarations, Vector result)
        throws IOException {
            switch (type) {
                case BYTE   : return new Byte(in.readByte());
                case UBYTE    : {
                    int b = in.read();
                    if (b == -1) throw new EOFException();
                    return new Integer(b);
                }
                case SHORT   : return new Short(in.readShort());
                case USHORT  : return new Integer(in.readShort() & 0xffff);
                case INT     : return new Integer(in.readInt());
                case UINT    : return new Long(in.readInt() & (long) 0xffffffff);
                case LONG    : return new Long(in.readLong());
                case FLOAT   : return new Float(in.readFloat());
                case DOUBLE  : return new Double(in.readDouble());
                case EXTENDED  : {
                    byte[] bits = new byte[10];
                    in.readFully(bits);
                    return new ExtendedReal(bits);
                }
                case CHAR    : return new Character(in.readChar());
                case CSTRING : {
                    StringBuffer buf = new StringBuffer();
                    int ch;
                    while ((ch = in.read()) > 0) {
                        buf.append((char) ch);
                    }
                    return buf.toString();
                }
                case PSTRING : {
                    int size = in.read();
                    if (size == 0) {
                        size = in.read();
                        in.read();
                        in.read();
                    }
                    byte[] bytes = new byte[size];
                    in.readFully(bytes);
                    return new String(bytes);
                }
                case MAGIC : {
                    int magic = in.readInt();
                    byte[] bytes = new byte[4];
                    bytes[0] = (byte)(magic >>> 24);
                    bytes[1] = (byte)(magic >>> 16);
                    bytes[2] = (byte)(magic >>>  8);
                    bytes[3] = (byte)(magic >>>  0);
                    return new String(bytes);
                }
                case MAC_TIMESTAMP : {
                    long secondsSince1904 = ((long) in.readInt()) & 0xffffffffL;
                    Calendar cal = Calendar.getInstance();
                    cal.set(1904, 0, 1);
                    return DateFormat.getInstance().format(
                    new Date(cal.getTime().getTime() + secondsSince1904 * 1000)
                    );
                }
                case TYPEDEF_TYPE : {
                    TypedefDeclaration typedefDeclaration = (TypedefDeclaration) declarations.typedefs.get(typedef);
                    if (typedefDeclaration == null)
                        throw new IOException("typedef not found for:"+typedef);
                    return typedefDeclaration.read(in, declarations, result);
                }
                case SHORTLE   : {
                    int b0 = in.read();
                    int b1 = in.read();
                    if (b1 == -1) throw new EOFException();
                    
                    return new Short((short) (b0 & 0xff | (b1 & 0xff) << 8));
                }
                case USHORTLE  : {
                    int b0 = in.read();
                    int b1 = in.read();
                    if (b1 == -1) throw new EOFException();
                    
                    return new Integer(b0 & 0xff | ((b1 & 0xff) << 8));
                }
                case INTLE    :  {
                    int b0 = in.read();
                    int b1 = in.read();
                    int b2 = in.read();
                    int b3 = in.read();
                    if (b3 == -1) throw new EOFException();
                    
                    return new Integer(b0 & 0xff | ((b1 & 0xff) << 8)
                    | ((b2 & 0xff) << 16)  | ((b3 & 0xff) << 24));
                }
                case UINTLE    : {
                    int b0 = in.read();
                    int b1 = in.read();
                    int b2 = in.read();
                    int b3 = in.read();
                    if (b3 == -1) throw new EOFException();
                    
                    return new Long(((long) (b0 & 0xff | ((b1 & 0xff) << 8)
                    | ((b2 & 0xff) << 16)  | ((b3 & 0xff) << 24))) & 0x00ffffffff) ;
                }
                case LONGLE    : {
                    int b0 = in.read();
                    int b1 = in.read();
                    int b2 = in.read();
                    int b3 = in.read();
                    int b4 = in.read();
                    int b5 = in.read();
                    int b6 = in.read();
                    int b7 = in.read();
                    if (b7 == -1) throw new EOFException();
                    
                    return new Long(
                    (long) b0 & 0xff
                    | ((long) (b1 & 0xff) << 8)
                    | ((long) (b2 & 0xff) << 16)
                    | ((long) (b3 & 0xff) << 24)
                    | ((long) (b4 & 0xff) << 32)
                    | ((long) (b5 & 0xff) << 40)
                    | ((long) (b6 & 0xff) << 48)
                    | ((long) (b7 & 0xff) << 56)
                    );
                }
                default :
                    throw new InternalError("invalid type:"+type);
            }
        }
    }
    /**
     * ArraySize expression.
     *
     * <pre><code>
     * ArraySize ::= () | intLiteral ["-" intLiteral]
     * </code></pre>
     */
    protected static class ArraySize {
        public final static int REMAINDER = -1;
        public final static int LITERAL = 0;
        public final static int VARIABLE = 1;
        public int type = LITERAL;
        public int size;
        public String variable;
        public int offset;
        
        public ArraySize(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() == ']') {
                scanner.pushBack();
                type = REMAINDER;
            } else if (scanner.ttype == StreamTokenizer.TT_WORD) {
                variable = scanner.sval;
                type = VARIABLE;
                if (scanner.nextToken() == '-') {
                    offset = new IntLiteral(scanner).intValue();
                } else {
                    scanner.pushBack();
                }
            } else {
                scanner.pushBack();
                size = new IntLiteral(scanner).intValue();
                type = LITERAL;
            }
        }
        public int getArraySize(Declarations declarations, Vector result) {
            switch (type) {
                case LITERAL :
                    return size;
                case VARIABLE :
                    for (int i=result.size() - 1; i > -1; i--) {
                        StructTableModel.Value value = (StructTableModel.Value) result.elementAt(i);
                        if (value.declaration.equals(variable)) {
                            return ((Number) value.value).intValue() - offset;
                        }
                    }
                    throw new InternalError("Invalid ArraySize variable:"+variable);
                case REMAINDER :
                    return -1;
                default :
                    throw new InternalError("Invalid ArraySize type:"+type);
            }
        }
    }
    
    /**
     * IntLiteral expression.
     *
     * <pre><code>
     * IntLiteral ::= intLiteral | hexLiteral
     * </code></pre>
     */
    protected static class IntLiteral {
        public int value;
        
        public IntLiteral(StreamTokenizer scanner)
        throws IOException, ParseException {
            if (scanner.nextToken() != StreamTokenizer.TT_NUMBER) {
                throw new ParseException(errorMsg("IntLiteral: numeric value expected",scanner));
            }
            value = (int) scanner.nval;
            if (scanner.nval == 0.0) {
                if (scanner.nextToken() == StreamTokenizer.TT_WORD && scanner.sval.startsWith("x")) {
                    value = Integer.valueOf(scanner.sval.substring(1), 16).intValue();
                } else {
                    scanner.pushBack();
                }
            }
        }
        public int intValue() {
            return value;
        }
    }
    /**
     * MagicOrIntLiteral expression.
     *
     * <pre><code>
     * MagicOrIntLiteral ::= magicLiteral | intLiteral | hexLiteral
     * </code></pre>
     */
    protected static class MagicOrIntLiteral {
        public int intValue;
        public String magicValue;
        
        public MagicOrIntLiteral(StreamTokenizer scanner)
        throws IOException, ParseException {
            switch (scanner.nextToken()) {
                case StreamTokenizer.TT_NUMBER :
                    intValue = (int) scanner.nval;
                    if (scanner.nval == 0.0) {
                        if (scanner.nextToken() == StreamTokenizer.TT_WORD && scanner.sval.startsWith("x")) {
                            intValue = Integer.valueOf(scanner.sval.substring(1), 16).intValue();
                        } else {
                            scanner.pushBack();
                        }
                    }
                    break;
                case '"' :
                    if (scanner.sval.length() != 4) {
                        throw new ParseException(errorMsg("MagicOrIntLiteral: magic with length of 4 characters expected",scanner));
                    }
                    magicValue = scanner.sval;
                    intValue = toInt(magicValue);
                    break;
                default :
                    throw new ParseException(errorMsg("MagicOrIntLiteral: numeric value expected",scanner));
            }
        }
        public boolean isMagic() {
            return magicValue != null;
        }
        public int intValue() {
            return intValue;
        }
        public String stringValue() {
            return (magicValue == null) ? Integer.toString(intValue) : magicValue;
        }
        public static int toInt(String magic) {
            return ((magic.charAt(0) & 0xff) << 24)
            | ((magic.charAt(1) & 0xff) << 16)
            | ((magic.charAt(2) & 0xff) << 8)
            | ((magic.charAt(3) & 0xff) << 0);
        }
        public static String toMagic(int value) {
            byte[] buf = new byte[4];
            buf[0] = (byte) ((value & 0xff000000) >>> 24);
            buf[1] = (byte) ((value & 0xff0000) >>> 16);
            buf[2] = (byte) ((value & 0xff00) >>> 8);
            buf[3] = (byte) (value & 0xff);
            try {
                return new String(buf, "ASCII");
            } catch (UnsupportedEncodingException e) {
                InternalError error = new InternalError(e.getMessage());
                error.initCause(e);
                throw error;
            }
        }
    }
}
